<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;
use Carbon\Carbon;

class VerifyPasscode
{
    protected $timeout = 600;

    public function handle(Request $request, Closure $next): Response
    {
        // Check if the user is authenticated
        if (!auth()->check()) {
            return redirect()->route('login');
        }

        // Check if the user is an admin
        if (auth()->user()->is_admin) {
            // Optional: Redirect admins to the admin dashboard or allow further processing
            return redirect()->route('admin.dashboard');
        }

        // Ensure the user has set up a passcode
        if (!auth()->user()->hasPasscodeSetup()) {
            return redirect()->route('passcode.setup')
                ->with('warning', 'Please set up your passcode to continue.');
        }

        // Check the session for the last passcode verification time
        $lastVerified = session('passcode_verified_at');

        if (!$lastVerified || Carbon::parse($lastVerified)->addSeconds($this->timeout)->isPast()) {
            // Clear session data related to passcode
            session()->forget('passcode_verified_at');
            session()->forget('passcode_verified');

            return redirect()->route('passcode.show')
                ->with('warning', 'Please re-enter your passcode.');
        }

        // Update the session timestamp for passcode verification
        session()->put('passcode_verified_at', now());

        return $next($request);
    }
}
