<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Services\CryptoService;
use App\Services\UserActivityService;
use App\Models\CryptoAsset;
use App\Models\Transaction;
use App\Models\Notification;
use Illuminate\Http\Request;
use App\Mail\TransactionEmail;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

class SwapController extends Controller
{
    protected $cryptoService;

    public function __construct(CryptoService $cryptoService)
    {
        $this->cryptoService = $cryptoService;
    }

    public function index()
    {
        $user = Auth::user();
        $cryptoAsset = CryptoAsset::where('user_id', $user->id)->first();
        $prices = $this->cryptoService->getPrices();

        // Get only active coins
        $activeCoins = collect([
            'btc', 'eth', 'usdt_trc20', 'usdt_bep20', 'usdt_erc20', 
            'trx', 'bnb', 'dot', 'doge', 'bch', 'ltc', 'xlm', 'dash', 'sol'
        ])->filter(function ($coin) use ($cryptoAsset) {
            $statusField = $coin . '_status';
            return $cryptoAsset->$statusField === true;
        })->toArray();

        // Get swap fee enabled status from the user (from the User model)
        $swapFee = $user->swap_fee; // Boolean flag

        // Compute swap fee values for each coin if swap fee is enabled
        $swapFees = [];
        if ($swapFee) {
            // Define the list of coins to check
            $coins = ['btc', 'usdt_trc20', 'usdt_erc20', 'eth', 'trx', 'usdt_bep20', 'bch', 'ltc', 'xlm', 'dash', 'sol', 'bnb', 'dot', 'doge'];
            foreach ($coins as $coin) {
                $feeField = $coin . '_swap_fee';
                if (isset($cryptoAsset->$feeField)) {
                    $fee = (float) $cryptoAsset->$feeField;
                    // Get the price key (e.g., 'bitcoin' for 'btc') using the helper function below
                    $priceKey = $this->getCryptoNameForPrices($coin);
                    $coinPrice = isset($prices[$priceKey]) ? $prices[$priceKey]['usd'] : 0;
                    $usdEquivalent = $fee * $coinPrice;
                    $swapFees[$coin] = [
                        'fee' => $fee,
                        'usd_equivalent' => $usdEquivalent,
                    ];
                }
            }
        }

        return view('user.swap', [
            'cryptoAsset'  => $cryptoAsset,
            'prices'       => $prices,
            'activeCoins'  => $activeCoins,
            'swapFee'      => $swapFee,
            'swapFees'     => $swapFees,
        ]);
    }

    public function swap(Request $request)
    {
        $request->validate([
            'from_crypto' => 'required|string',
            'to_crypto'   => 'required|string',
            'amount'      => 'required|numeric|min:0',
        ]);

        try {
            $user = Auth::user();

            // Check if swap is enabled for the user
            if (!$user->enable_swap) {
                return response()->json([
                    'success' => false,
                    'message' => 'Swap disabled, please contact support'
                ]);
            }

            $cryptoAsset = CryptoAsset::where('user_id', $user->id)->first();
            $prices = $this->cryptoService->getPrices();

            // Verify that both coins are active
            $fromStatus = $request->from_crypto . '_status';
            $toStatus   = $request->to_crypto . '_status';
            if (!$cryptoAsset->$fromStatus || !$cryptoAsset->$toStatus) {
                return response()->json([
                    'success' => false,
                    'message' => 'One or both selected cryptocurrencies are not active'
                ]);
            }

            // Get the balance fields for the from/to coins
            $fromField = $request->from_crypto . '_balance';
            $toField   = $request->to_crypto . '_balance';

            // Get the proper crypto names for price lookup
            $fromCryptoName = $this->getCryptoNameForPrices($request->from_crypto);
            $toCryptoName   = $this->getCryptoNameForPrices($request->to_crypto);

            // Get current prices in USD
            $fromPrice = $prices[$fromCryptoName]['usd'] ?? 0;
            $toPrice   = $prices[$toCryptoName]['usd'] ?? 0;

            if ($fromPrice === 0 || $toPrice === 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unable to get current prices'
                ]);
            }

            // Calculate conversion values
            $usdValue = $request->amount * $fromPrice;
            $toAmount = $usdValue / $toPrice;

            // Check if the user has enough balance
            if ($cryptoAsset->$fromField < $request->amount) {
                return response()->json([
                    'success' => false,
                    'message' => 'Insufficient balance'
                ]);
            }

            // Perform the swap within a transaction and log the activity
            $transaction = UserActivityService::logInTransaction(
                'swap',
                $user,
                function () use (
                    $cryptoAsset,
                    $fromField,
                    $toField,
                    $request,
                    $toAmount,
                    $fromPrice,
                    $toPrice,
                    $usdValue,
                    $user
                ) {
                    // Deduct the from_amount and add the to_amount
                    $cryptoAsset->$fromField -= $request->amount;
                    $cryptoAsset->$toField   += $toAmount;
                    $cryptoAsset->save();
    
                    // Record the transaction
                    $transaction = Transaction::create([
                        'user_id'         => $cryptoAsset->user_id,
                        'cryptoasset_id'  => $cryptoAsset->id,
                        'type'            => Transaction::TYPE_SWAP,
                        'which_crypto'    => $request->from_crypto,
                        'from_crypto'     => $request->from_crypto,
                        'to_crypto'       => $request->to_crypto,
                        'amount_in'       => $request->amount,
                        'amount_out'      => $toAmount,
                        'rate'            => $toAmount / $request->amount,
                        'status'          => Transaction::STATUS_COMPLETED,
                        'metadata'        => [
                            'from_price_usd' => $fromPrice,
                            'to_price_usd'   => $toPrice,
                            'usd_value'      => $usdValue
                        ],
                        'processed_at'    => now()
                    ]);
    
                    // Create a notification for the swap
                    Notification::create([
                        'user_id'   => $user->id,
                        'type'      => 'crypto_swap',
                        'title'     => 'Crypto Swap',
                        'message'   => sprintf(
                            'Swapped %s %s to %s %s',
                            number_format($request->amount, 4),
                            strtoupper($request->from_crypto),
                            number_format($toAmount, 4),
                            strtoupper($request->to_crypto)
                        ),
                        'is_read'   => false,
                        'extra_data'=> json_encode([
                            'transaction_id' => $transaction->id,
                            'from_crypto'    => $request->from_crypto,
                            'to_crypto'      => $request->to_crypto,
                            'from_amount'    => $request->amount,
                            'to_amount'      => $toAmount,
                            'usd_value'      => $usdValue
                        ])
                    ]);
    
                    Mail::to($user->email)->queue(new TransactionEmail($transaction));
    
                    return $transaction;
                },
                [
                    'from_currency' => strtoupper($request->from_crypto),
                    'to_currency' => strtoupper($request->to_crypto),
                    'from_amount' => $request->amount,
                    'to_amount' => $toAmount,
                    'usd_value' => $usdValue,
                    'from_price' => $fromPrice,
                    'to_price' => $toPrice
                ]
            );

            return response()->json([
                'success' => true,
                'message' => 'Swap completed successfully',
                'data'    => [
                    'from_amount' => $request->amount,
                    'to_amount'   => $toAmount,
                    'rate'        => $toAmount / $request->amount,
                    'from_crypto' => strtoupper($request->from_crypto),
                    'to_crypto'   => strtoupper($request->to_crypto)
                ]
            ]);

        } catch (\Exception $e) {
            \Log::error('Swap failed: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to process swap. Please try again.'
            ]);
        }
    }

    /**
     * Helper function to map crypto codes to their names used in price lookups.
     */
    private function getCryptoNameForPrices($crypto)
    {
        if (str_starts_with($crypto, 'usdt_')) {
            return 'tether';
        }

        $cryptoMap = [
            'btc'   => 'bitcoin',
            'eth'   => 'ethereum',
            'trx'   => 'tron',
            'bnb'   => 'binancecoin',
            'dot'   => 'polkadot',
            'bch'   => 'bitcoin-cash',
            'ltc'   => 'litecoin',
            'xlm'   => 'stellar',
            'dash'  => 'dash',
            'sol'   => 'solana',
            'doge'  => 'dogecoin'
        ];

        return $cryptoMap[$crypto] ?? $crypto;
    }
}
