<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\CryptoAsset;
use App\Services\CryptoService;
use Illuminate\Support\Facades\Auth;

class BuyController extends Controller
{
    private $cryptoService;

    public function __construct(CryptoService $cryptoService)
    {
        $this->cryptoService = $cryptoService;
    }

    public function index()
    {
        $user = Auth::user();
        $cryptoAsset = CryptoAsset::where('user_id', $user->id)->first();
        
        if (!$cryptoAsset) {
            return redirect()->route('dashboard')->with('error', 'No crypto assets found');
        }

        $assets = $this->mapCryptoAssets($cryptoAsset);

        return view('user.buy.index', [
            'assets' => $assets
        ]);
    }

    private function formatAsset($symbol, $name, $coinId, $balance, $price, $change, $network = null, $networkSymbol = null)
    {
        return [
            'symbol' => $symbol,
            'name' => $name . ($network ? " ($network)" : ''),
            'balance' => $balance,
            'value' => $balance * $price,
            'price' => $price,
            'change' => $change < 0 ? number_format($change, 2) : '+' . number_format($change, 2),
            'icon_url' => $this->cryptoService->getIconUrl($coinId),
            'network_url' => $networkSymbol ? $this->cryptoService->getIconUrl($networkSymbol) : null,
            'network' => $network
        ];
    }

    private function mapCryptoAssets(CryptoAsset $cryptoAsset)
    {
        $prices = $this->cryptoService->getPrices();
        $assets = [];

        // Bitcoin
        if ($cryptoAsset->btc_status) {
            $assets[] = $this->formatAsset(
                'BTC',
                'Bitcoin',
                'bitcoin',
                $cryptoAsset->btc_balance,
                $prices['bitcoin']['usd'] ?? 0,
                $prices['bitcoin']['usd_24h_change'] ?? 0
            );
        }

        // USDT with different networks
        $usdtNetworks = [
            'trc20' => ['status' => 'usdt_trc20_status', 'balance' => 'usdt_trc20_balance', 'network' => 'TRC20', 'symbol' => 'tron'],
            'bep20' => ['status' => 'usdt_bep20_status', 'balance' => 'usdt_bep20_balance', 'network' => 'BNB', 'symbol' => 'binancecoin'],
            'erc20' => ['status' => 'usdt_erc20_status', 'balance' => 'usdt_erc20_balance', 'network' => 'ERC20', 'symbol' => 'ethereum']
        ];

        foreach ($usdtNetworks as $network => $config) {
            if ($cryptoAsset->{$config['status']}) {
                $assets[] = $this->formatAsset(
                    'USDT',
                    'Tether',
                    'tether',
                    $cryptoAsset->{$config['balance']},
                    $prices['tether']['usd'] ?? 0,
                    $prices['tether']['usd_24h_change'] ?? 0,
                    $config['network'],
                    $config['symbol']
                );
            }
        }

        // Other cryptocurrencies mapping
        $cryptoMapping = [
            'eth' => ['symbol' => 'ETH', 'name' => 'Ethereum', 'id' => 'ethereum'],
            'trx' => ['symbol' => 'TRX', 'name' => 'Tron', 'id' => 'tron'],
            'bnb' => ['symbol' => 'BNB', 'name' => 'Binance Coin', 'id' => 'binancecoin'],
            'dot' => ['symbol' => 'DOT', 'name' => 'Polkadot', 'id' => 'polkadot'],
            'bch' => ['symbol' => 'BCH', 'name' => 'Bitcoin Cash', 'id' => 'bitcoin-cash'],
            'ltc' => ['symbol' => 'LTC', 'name' => 'Litecoin', 'id' => 'litecoin'],
            'xlm' => ['symbol' => 'XLM', 'name' => 'Stellar', 'id' => 'stellar'],
            'dash' => ['symbol' => 'DASH', 'name' => 'Dash', 'id' => 'dash'],
            'sol' => ['symbol' => 'SOL', 'name' => 'Solana', 'id' => 'solana'],
            'doge' => ['symbol' => 'DOGE', 'name' => 'Dogecoin', 'id' => 'dogecoin']
        ];

        foreach ($cryptoMapping as $key => $crypto) {
            $statusField = "{$key}_status";
            $balanceField = "{$key}_balance";
            
            if ($cryptoAsset->$statusField) {
                $assets[] = $this->formatAsset(
                    $crypto['symbol'],
                    $crypto['name'],
                    $crypto['id'],
                    $cryptoAsset->$balanceField,
                    $prices[$crypto['id']]['usd'] ?? 0,
                    $prices[$crypto['id']]['usd_24h_change'] ?? 0
                );
            }
        }

        return $assets;
    }

    public function details($symbol, $network = 'native')
    {
        $user = Auth::user();
        $cryptoAsset = CryptoAsset::where('user_id', $user->id)->first();
        
        if (!$cryptoAsset) {
            return redirect()->route('dashboard')->with('error', 'No crypto assets found');
        }

        $prices = $this->cryptoService->getPrices();
        $asset = null;

        // Handle USDT with different networks
        if (strtoupper($symbol) === 'USDT' && $network !== 'native') {
            $networkMap = ['BNB' => 'BEP20'];
            $network = $networkMap[$network] ?? $network;
            
            $balanceField = 'usdt_' . strtolower($network) . '_balance';
            $statusField = 'usdt_' . strtolower($network) . '_status';
            
            if ($cryptoAsset->$statusField) {
                $networkSymbol = match(strtolower($network)) {
                    'trc20' => 'tron',
                    'bep20' => 'binancecoin',
                    'erc20' => 'ethereum',
                    default => null
                };

                $asset = $this->formatAsset(
                    'USDT',
                    'Tether',
                    'tether',
                    $cryptoAsset->$balanceField,
                    $prices['tether']['usd'] ?? 0,
                    $prices['tether']['usd_24h_change'] ?? 0,
                    strtoupper($network),
                    $networkSymbol
                );
            }
        }
        // Handle other assets
        else {
            $balanceField = strtolower($symbol) . '_balance';
            if ($cryptoAsset->$balanceField) {
                $coinId = $this->getCoinGeckoId($symbol);
                if (isset($prices[$coinId])) {
                    $asset = $this->formatAsset(
                        strtoupper($symbol),
                        ucfirst($symbol),
                        $coinId,
                        $cryptoAsset->$balanceField,
                        $prices[$coinId]['usd'] ?? 0,
                        $prices[$coinId]['usd_24h_change'] ?? 0
                    );
                }
            }
        }

        if (!$asset) {
            return redirect()->route('dashboard')->with('error', 'Asset not found');
        }

        return view('user.buy.details', [
            'asset' => $asset
        ]);
    }

    private function getCoinGeckoId($symbol)
    {
        $symbolToId = [
            'btc' => 'bitcoin',
            'eth' => 'ethereum',
            'usdt' => 'tether',
            'trx' => 'tron',
            'bnb' => 'binancecoin',
            'dot' => 'polkadot',
            'bch' => 'bitcoin-cash',
            'ltc' => 'litecoin',
            'xlm' => 'stellar',
            'dash' => 'dash',
            'sol' => 'solana',
            'doge' => 'dogecoin'
        ];

        return $symbolToId[strtolower($symbol)] ?? strtolower($symbol);
    }
}